<?php
/**
 * Copyright 2017 Adobe
 * All Rights Reserved.
 */
namespace Magento\Backend\Block;

use Magento\Backend\Model\Menu\Item;
use Magento\Framework\Escaper;

/**
 * Class AnchorRenderer
 */
class AnchorRenderer
{
    /**
     * @var MenuItemChecker
     */
    private $menuItemChecker;

    /**
     * @var Escaper
     */
    private $escaper;

    /**
     * @param MenuItemChecker $menuItemChecker
     * @param Escaper $escaper
     */
    public function __construct(
        MenuItemChecker $menuItemChecker,
        Escaper $escaper
    ) {
        $this->menuItemChecker = $menuItemChecker;
        $this->escaper = $escaper;
    }

    /**
     * Render menu item anchor.
     *
     *  It is used in backend menu to render anchor menu.
     *
     * @param Item|false $activeItem Can be false if menu item is inaccessible
     * but was triggered directly using controller. It is a legacy code behaviour.
     * @param Item $menuItem
     * @param int $level
     * @return string
     */
    public function renderAnchor($activeItem, Item $menuItem, $level)
    {
        if ($level == 1 && $menuItem->getUrl() == '#') {
            $output = '';
            if ($menuItem->hasChildren()) {
                $output = '<strong class="submenu-group-title" role="presentation">'
                    . '<span>' . $this->escaper->escapeHtml(__($menuItem->getTitle())) . '</span>'
                    . '</strong>';
            }
        } else {
            $target = $menuItem->getTarget() ? ('target=' . $menuItem->getTarget()) : '';
            $output = '<a href="' . $menuItem->getUrl() . '" ' . $target . ' ' . $this->_renderItemAnchorTitle(
                $menuItem
            ) . $this->_renderItemOnclickFunction(
                $menuItem
            ) . ' class="' . ($this->menuItemChecker->isItemActive($activeItem, $menuItem, $level) ? '_active' : '')
                . '">' . '<span>' . $this->escaper->escapeHtml(__($menuItem->getTitle()))
                . '</span>' . '</a>';
        }

        return $output;
    }

    /**
     * Render menu item anchor title
     *
     * @param Item $menuItem
     * @return string
     */
    private function _renderItemAnchorTitle($menuItem)
    {
        return $menuItem->hasTooltip() ? 'title="' . __($menuItem->getTooltip()) . '"' : '';
    }

    /**
     * Render menu item onclick function
     *
     * @param Item $menuItem
     * @return string
     */
    private function _renderItemOnclickFunction($menuItem)
    {
        return $menuItem->hasClickCallback() ? ' onclick="' . $menuItem->getClickCallback() . '"' : '';
    }
}
