<?php
/**
 * Copyright 2019 Adobe
 * All Rights Reserved.
 */
declare(strict_types=1);

namespace Magento\InventoryInStorePickupSales\Model;

use Magento\InventoryInStorePickupSales\Model\Order\IsFulfillable;
use Magento\InventoryInStorePickupSalesApi\Model\IsOrderReadyForPickupInterface;
use Magento\Sales\Api\Data\OrderInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\Order;

/**
 * Check if order can be shipped and the pickup location has enough QTY
 */
class IsOrderReadyForPickup implements IsOrderReadyForPickupInterface
{
    /**
     * @var IsFulfillable
     */
    private $isFulfillable;

    /**
     * @var OrderRepositoryInterface
     */
    private $orderRepository;

    /**
     * @param IsFulfillable $isFulfillable
     * @param OrderRepositoryInterface $orderRepository
     */
    public function __construct(
        IsFulfillable $isFulfillable,
        OrderRepositoryInterface $orderRepository
    ) {
        $this->isFulfillable = $isFulfillable;
        $this->orderRepository = $orderRepository;
    }

    /**
     * Check if order can be shipped and the pickup location has enough QTY.
     *
     * @param int $orderId
     * @return bool
     */
    public function execute(int $orderId): bool
    {
        $order = $this->orderRepository->get($orderId);

        return $this->canShip($order) && $this->isFulfillable->execute($order);
    }

    /**
     * Retrieve order shipment availability.
     *
     * @param OrderInterface $order
     * @return bool
     */
    private function canShip(OrderInterface $order): bool
    {
        if ($order instanceof Order) {
            return $order->canShip();
        }

        return true;
    }
}
