<?php
/**
 * Copyright 2019 Adobe
 * All Rights Reserved.
 */
declare(strict_types=1);

namespace Magento\InventoryInStorePickupShippingApi\Model\Carrier\Validation;

use Magento\Framework\Validation\ValidationResult;
use Magento\Framework\Validation\ValidationResultFactory;
use Magento\Quote\Model\Quote\Address\RateRequest;

/**
 * @inheritdoc
 */
class RequestValidatorChain implements RequestValidatorInterface
{
    /**
     * @var ValidationResultFactory
     */
    private $validationResultFactory;

    /**
     * @var RequestValidatorInterface[]
     */
    private $validators;

    /**
     * @param ValidationResultFactory $validationResultFactory
     * @param array $validators
     */
    public function __construct(
        ValidationResultFactory $validationResultFactory,
        array $validators = []
    ) {
        $this->validationResultFactory = $validationResultFactory;

        foreach ($validators as $validator) {
            if (!$validator instanceof RequestValidatorInterface) {
                throw new \InvalidArgumentException(
                    sprintf(
                        'In-Store Pickup Carrier Rate Request Validator must implement %s.' .
                        '%s has been received instead.',
                        RequestValidatorInterface::class,
                        get_class($validator)
                    )
                );
            }
        }
        $this->validators = $validators;
    }

    /**
     * @inheritdoc
     */
    public function validate(RateRequest $rateRequest): ValidationResult
    {
        $errors = [];
        /** @var RequestValidatorInterface $validator */
        foreach ($this->validators as $validator) {
            $validationResult = $validator->validate($rateRequest);

            if (!$validationResult->isValid()) {
                //phpcs:ignore Magento2.Performance.ForeachArrayMerge
                $errors = array_merge($errors, $validationResult->getErrors());
            }
        }

        return $this->validationResultFactory->create(['errors' => $errors]);
    }
}
